/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/


#include "mx__shim.h"
#include "mx_io.h"
#include "mx_io_impl.h"
#include "mx__fops.h"
#include "mx__udrv_fops.h"
#include "mx__regcache.h"
#include "mx_arch_comm.h"
#include "mx_raw.h"
#include <sys/mman.h>
#include <unistd.h>
#include <sys/un.h>

int
mx__udrv_open(int unit, int endpoint, mx_endpt_handle_t *handle)
{
  int rc;
  int sock;
  extern int errno;
  struct sockaddr_un un;
  
  if (unit > 0)
    return MX_FAILURE;
  mx_sprintf(un.sun_path, MX_IOCTL_SOCK);
  sock = socket(AF_UNIX, SOCK_STREAM, 0);
  if (sock < 0)
    return errno == ENOENT ? MX_NO_DEV 
      :  errno == EPERM || errno == EACCES ? MX_NO_PERM
      :  MX_FAILURE;
  un.sun_family = AF_UNIX;
  rc = connect(sock,(struct sockaddr*)&un,sizeof(un));
  if (rc < 0) {
    close(sock);
    return MX_FAILURE;
  }
  *handle = sock;
  return MX_SUCCESS;
}

static uint32_t mx_address_space;

int
mx__udrv_ioctl(mx_endpt_handle_t handle, int cmd, void *buf, size_t bufsize)
{
  struct iobuf {
    uint32_t cmd_status;
    char data[0];
  } *iobuf = 0;
  mx_nic_id_hostname_t *nic_host = buf;
  mx_set_hostname_t *set_host = buf;
  mx_raw_send_t *rsend = buf;
  mx_raw_next_event_t *rnext = buf;
  mx_get_counters_strings_t *s_ctr = buf;
  mx_get_logging_t *glog = buf;
  mx_get_logging_strings_t *s_log = buf;
  mx_get_route_table_t *tbl = buf;
  mx_set_route_t *sroute = buf;
  mx_get_eeprom_string_t *geeprom = buf;
  mx_reg_t rdma;
  int isize;
  int rc;

  /* handle scatter/gather cases of the driver/lib interface */
  switch (cmd) {
  case MX_REGISTER:
    isize = bufsize;
    rdma = *(mx_reg_t *)buf;
    mx_always_assert(rdma.nsegs == 1);
    mx_always_assert(rdma.segs.vaddr >= (size_t)mx__init_brk);
    rdma.segs.vaddr = ((uint64_t)mx_address_space << 32) +
      0x80000000 + rdma.segs.vaddr - (size_t)mx__init_brk;
    buf = &rdma;
    break;
  case MX_GET_ROUTE_TABLE:
    isize = bufsize;
    bufsize = 16384;
    break;
  case MX_GET_PRODUCT_CODE:
  case MX_GET_PART_NUMBER:
    isize = bufsize;
    bufsize = bufsize + 256;
    break;
  case MX_GET_COUNTERS:
    isize = bufsize;
    bufsize = 16384;
    break;
  case MX_GET_LOGGING:
    isize = bufsize;
    bufsize += glog->size;
    break;
  case MX_GET_MAPPER_MSGBUF:
  case MX_GET_MAPPER_MAPBUF:
    isize = 0;
    bufsize = 32768;
    break;
  case MX_GET_PEER_TABLE:
    bufsize = 16384;
    isize = 0;
    break;
  case MX_NIC_ID_TO_HOSTNAME:
  case MX_HOSTNAME_TO_NIC_ID:
    bufsize += nic_host->len;
    isize = bufsize;
    iobuf = alloca(bufsize + sizeof(*iobuf));
    memcpy(iobuf->data, buf, sizeof(*nic_host));
    memcpy(iobuf->data+sizeof(*nic_host),(void*)(uintptr_t)nic_host->va, nic_host->len);
    break;
  case MX_SET_HOSTNAME:
    bufsize += set_host->len;
    isize = bufsize;
    iobuf = alloca(bufsize + sizeof(*iobuf));
    memcpy(iobuf->data, buf, sizeof(*set_host));
    memcpy(iobuf->data+sizeof(*set_host),(void*)(uintptr_t)set_host->va, set_host->len);
    break;
  case MX_SET_ROUTE:
    bufsize += sroute->route_length;
    isize = bufsize;
    iobuf = alloca(bufsize + sizeof(*iobuf));
    memcpy(iobuf->data, buf, sizeof(*sroute));
    memcpy(iobuf->data+sizeof(*sroute),(void*)(uintptr_t)sroute->route_pointer, sroute->route_length);
    break;
  case MX_RAW_SEND:
    bufsize += rsend->buffer_length + rsend->route_length;
    isize = bufsize;
    iobuf = alloca(bufsize + sizeof(*iobuf));
    memcpy(iobuf->data, buf, sizeof(*rsend));
    memcpy(iobuf->data+sizeof(*rsend),(void*)(uintptr_t)rsend->route_pointer, rsend->route_length);
    memcpy(iobuf->data+sizeof(*rsend)+rsend->route_length,
	   (void*)(uintptr_t)rsend->data_pointer, rsend->buffer_length);
    break;
  case MX_RAW_GET_NEXT_EVENT:
    isize = bufsize;
    bufsize += 2048;
    break;
  case MX_GET_COUNTERS_STRINGS:
    isize = bufsize;
    bufsize += s_ctr->count == 0 ? 0 : (s_ctr->count - 1) * 128;
    break;
  case MX_GET_LOGGING_STRINGS:
    isize = bufsize;
    bufsize += s_log->count == 0 ? 0 : (s_ctr->count - 1) * 128;
    break;
  default:
    isize = bufsize;
  }
  if (!iobuf) {
    iobuf = alloca(bufsize + sizeof(*iobuf));
    memcpy(iobuf->data, buf, isize);
  }
  iobuf->cmd_status = cmd;
  rc = udrv_send(handle,iobuf,bufsize + sizeof(*iobuf),0);
  mx_always_assert(rc == bufsize + sizeof(*iobuf));
  rc = udrv_recv(handle,iobuf,bufsize + sizeof(*iobuf),0);
  mx_always_assert(rc >= sizeof(*iobuf) && rc <= bufsize + sizeof(*iobuf));
  /* handle wierd driver/lib interface */
  if (iobuf->cmd_status == 0) {
    uint64_t old_field;
    switch (cmd) {
    case MX_GET_PRODUCT_CODE:
    case MX_GET_PART_NUMBER:
      strncpy((void*)(uintptr_t)geeprom->buffer, iobuf->data+sizeof(*geeprom), MX_MAX_STR_LEN);
      memcpy(geeprom, iobuf->data, sizeof(*geeprom));
      break;
    case MX_NIC_ID_TO_HOSTNAME:
    case MX_HOSTNAME_TO_NIC_ID:
      memcpy((void*)(uintptr_t)nic_host->va, iobuf->data+sizeof(*nic_host), nic_host->len);
      memcpy(nic_host, iobuf->data, sizeof(*nic_host));
      break;
    case MX_SET_HOSTNAME:
      memcpy((void*)(uintptr_t)set_host->va, iobuf->data+sizeof(*set_host), set_host->len);
      memcpy(set_host, iobuf->data, sizeof(*set_host));
      break;
    case MX_RAW_GET_NEXT_EVENT:
      old_field = rnext->recv_buffer;
      memcpy(rnext, iobuf->data, sizeof(*rnext));
      rnext->recv_buffer = old_field;
      if (rnext->status == MX_RAW_RECV_COMPLETE)
	memcpy((void*)(uintptr_t)rnext->recv_buffer, iobuf->data+sizeof(*rnext), rnext->recv_bytes);
      break;
    case MX_GET_ROUTE_TABLE:
      memcpy((char*)(uintptr_t)tbl->routes, iobuf->data+sizeof(*tbl), rc - sizeof(*iobuf) - sizeof(*tbl));
      break;
    case MX_GET_LOGGING:
      memcpy((char*)(uintptr_t)glog->buffer, iobuf->data+sizeof(*glog), rc - sizeof(*iobuf) - sizeof(*glog));
      break;
    default:
      memcpy(buf, iobuf->data, rc - sizeof(*iobuf));
    }
    rc = MX_SUCCESS;
  } else if (iobuf->cmd_status == ENOSPC) {
    rc = MX_NO_RESOURCES;
  } else {
    rc = MX_BAD_BAD_BAD;
  }
  if (rc)
    fprintf(stderr,"command %x returned errno=%d:%s\n", cmd, iobuf->cmd_status, strerror(iobuf->cmd_status));
  return rc;
}


void *
mx__udrv_mmap(void *start, size_t length, mx_endpt_handle_t handle, ptrdiff_t offset)
{
  mx_mmapio_t x;
  int status;
  int fd;
  char *ptr;
  unsigned long pad;
  memset(&x,0,sizeof(x));
  x.offset = (uint32_t)offset;
  x.len = (uint32_t)length;
  x.as = mx_address_space;
  status =  mx__ioctl(handle, MX_MMAP, &x, sizeof(x));
  if (status)
    return (void *) -1;
  mx_address_space = x.as;
  fd = open(x.outpath,O_RDWR);
  if (fd < 0)
    return (void *)-1;
  pad = 0;
  mx_always_assert(!start || pad == 0);
  ptr = mmap(start, x.len,PROT_READ|PROT_WRITE, 
	     MAP_SHARED | MAP_FILE | (start ? MAP_FIXED: 0), fd, x.offset - pad);
  close(fd);
  return (uintptr_t)ptr == -1 ? ptr : ptr + pad;
}
